/*
 * Sphere 6502 assembler
 * Copyright 2018 Eggbertx
 * adapted from the original by Stian Soreng @ www.6502asm.com
 * with some help from Fat Cerberus and rhuan
 * Released under the GNU General Public License
 * see http://gnu.org/licenses/gpl.html
 */
import { Console } from 'sphere-runtime';

Sphere.frameRate = 150;
var screen = Surface.Screen;
var kb = Keyboard.Default;
var scale = Surface.Screen.width/32;
var font = new Font("@/liberation_mono.rfn");
var MAX_MEM = ((32 * 32) - 1);
var codeCompiledOK = false;
var regA = 0;
var regX = 0;
var regY = 0;
var regP = 0;
var regPC = 0x600;
var regSP = 0x100;
var memory = new Array(0x600);
var runForever = false;
var labelIndex = [];
var labelPtr = 0;
var codeRunning = false;
var codeLen = 0;
var codeSrc = "";
var display = new Array(0x400); // 1024, 32*32
var drawDebugInfo = false;
var defaultCodePC = 0x600;

function colorArrayOf(name) {
	let sphereColor = Color.of(name);
	return [
		Math.floor(sphereColor.r * 255),
		Math.floor(sphereColor.g * 255),
		Math.floor(sphereColor.b * 255)
	];
}

var palette = [
	colorArrayOf("#000000"), colorArrayOf("#ffffff"), colorArrayOf("#880000"), colorArrayOf("#aaffee"),
	colorArrayOf("#cc44cc"), colorArrayOf("#00cc55"), colorArrayOf("#0000aa"), colorArrayOf("#eeee77"),
	colorArrayOf("#dd8855"), colorArrayOf("#664400"), colorArrayOf("#ff7777"), colorArrayOf("#333333"),
	colorArrayOf("#777777"), colorArrayOf("#aaff66"), colorArrayOf("#0088ff"), colorArrayOf("#bbbbbb")
];


var Opcodes = [
	/*Name, Imm,  ZP,   ZPX,  ZPY,  ABS,  ABSX, ABSY, INDX, INDY, SNGL, BRA */
	["ADC", 0x69, 0x65, 0x75, 0x00, 0x6d, 0x7d, 0x79, 0x61, 0x71, 0x00, 0x00],
	["AND", 0x29, 0x25, 0x35, 0x00, 0x2d, 0x3d, 0x39, 0x21, 0x31, 0x00, 0x00],
	["ASL", 0x00, 0x06, 0x16, 0x00, 0x0e, 0x1e, 0x00, 0x00, 0x00, 0x0a, 0x00],
	["BIT", 0x00, 0x24, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["BPL", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10],
	["BMI", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30],
	["BVC", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50],
	["BVS", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70],
	["BCC", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90],
	["BCS", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0],
	["BNE", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd0],
	["BEQ", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0],
	["CMP", 0xc9, 0xc5, 0xd5, 0x00, 0xcd, 0xdd, 0xd9, 0xc1, 0xd1, 0x00, 0x00],
	["CPX", 0xe0, 0xe4, 0x00, 0x00, 0xec, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["CPY", 0xc0, 0xc4, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["DEC", 0x00, 0xc6, 0xd6, 0x00, 0xce, 0xde, 0x00, 0x00, 0x00, 0x00, 0x00],
	["EOR", 0x49, 0x45, 0x55, 0x00, 0x4d, 0x5d, 0x59, 0x41, 0x51, 0x00, 0x00],
	["CLC", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00],
	["SEC", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00],
	["CLI", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0x00],
	["SEI", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x00],
	["CLV", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x00],
	["CLD", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x00],
	["SED", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00],
	["INC", 0x00, 0xe6, 0xf6, 0x00, 0xee, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00],
	["JMP", 0x00, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["JSR", 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["LDA", 0xa9, 0xa5, 0xb5, 0x00, 0xad, 0xbd, 0xb9, 0xa1, 0xb1, 0x00, 0x00],
	["LDX", 0xa2, 0xa6, 0x00, 0xb6, 0xae, 0x00, 0xbe, 0x00, 0x00, 0x00, 0x00],
	["LDY", 0xa0, 0xa4, 0xb4, 0x00, 0xac, 0xbc, 0x00, 0x00, 0x00, 0x00, 0x00],
	["LSR", 0x00, 0x46, 0x56, 0x00, 0x4e, 0x5e, 0x00, 0x00, 0x00, 0x4a, 0x00],
	["NOP", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xea, 0x00],
	["ORA", 0x09, 0x05, 0x15, 0x00, 0x0d, 0x1d, 0x19, 0x01, 0x11, 0x00, 0x00],
	["TAX", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaa, 0x00],
	["TXA", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00],
	["DEX", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xca, 0x00],
	["INX", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x00],
	["TAY", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x00],
	["TYA", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x98, 0x00],
	["DEY", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00],
	["INY", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x00],
	["ROR", 0x00, 0x66, 0x76, 0x00, 0x6e, 0x7e, 0x00, 0x00, 0x00, 0x6a, 0x00],
	["ROL", 0x00, 0x26, 0x36, 0x00, 0x2e, 0x3e, 0x00, 0x00, 0x00, 0x2a, 0x00],
	["RTI", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00],
	["RTS", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00],
	["SBC", 0xe9, 0xe5, 0xf5, 0x00, 0xed, 0xfd, 0xf9, 0xe1, 0xf1, 0x00, 0x00],
	["STA", 0x00, 0x85, 0x95, 0x00, 0x8d, 0x9d, 0x99, 0x81, 0x91, 0x00, 0x00],
	["TXS", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x00],
	["TSX", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xba, 0x00],
	["PHA", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00],
	["PLA", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00],
	["PHP", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00],
	["PLP", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00],
	["STX", 0x00, 0x86, 0x00, 0x96, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["STY", 0x00, 0x84, 0x94, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00],
	["---", 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00]
];

/*
 *  reset() - Reset CPU and memory.
 *
 */

function reset() {
	for(var y = 0; y < 32; y++) {
		for(var x = 0; x < 32; x++) {
			display[y * 32 + x] = palette[0];
		}
	}
	for(var x = 0; x < 0x600; x++) {
		// clear ZP, stack and screen
		memory[x] = 0x00;
	}
	regA = regX = regY = 0;
	defaultCodePC = regPC = 0x600;
	regSP = 0x100;
	regP = 0x20;
	runForever = false;
}


/*
 *  message() - Prints text in the message window
 *
 */

function message(text) {
	if(arguments.length == 2 && arguments[1] == true) Sphere.abort(text); // fatal error
	else console.log(text);
	
}

/*
 *  compileCode()
 * 
 *  "Compiles" the code into a string (global var compiledCode)
 * 
 */

function compileCode(file) {
	reset();
	codeSrc = FS.readFile(file);
	var code = codeSrc + "\n\n";
	var lines = code.split("\n");
	codeCompiledOK = true;
	labelIndex = [];
	labelPtr = 0;

	message("Indexing labels..");
	defaultCodePC = regPC = 0x600;
	for(var xc = 0; xc < lines.length; xc++) {
		if(!indexLabels(lines[xc])) {
			message("<b>Label already defined at line " + (xc + 1) + ":</b> " + lines[xc]);
			return false;
		}
	}

	var str = "Found " + labelIndex.length + " label";
	if(labelIndex.length != 1) str += "s";
	message(str + ".");

	defaultCodePC = regPC = 0x600;
	message("Compiling code..");

	for(var x = 0; x < lines.length; x++) {
		if(!compileLine(lines[x], x)) {
			codeCompiledOK = false;
			break;
		}
	}

	if(codeLen == 0) {
		codeCompiledOK = false;
		message("No code to run.");
	}

	if(codeCompiledOK) memory[defaultCodePC] = 0x00;
	else {
		message("<b>Syntax error line " + (x + 1) + ": " + lines[x] + "</b>");
		return;
	}

	updateDisplayFull();
	message("Code compiled successfully, " + codeLen + " bytes.");
}

/*
 *  indexLabels() - Pushes all labels to array.
 *
 */

function indexLabels(input) {
	// remove comments
	input = input.replace(new RegExp(/^(.*?);.*/), "$1");

	// trim line
	input = input.replace(new RegExp(/^\s+/), "");
	input = input.replace(new RegExp(/\s+$/), "");

	// Figure out how many bytes this instuction takes
	var thisPC = defaultCodePC;

	codeLen = 0;
	// defaultCodePC = 0x600;
	compileLine(input);
	regPC += codeLen;

	// Find command or label
	if(input.match(new RegExp(/^\w+:/))) {
		var label = input.replace(new RegExp(/(^\w+):.*$/), "$1");
		return pushLabel(label + "|" + thisPC);
	}
	return true;
}

/*
 *  pushLabel() - Push label to array. Return false if label already exists.
 * 
 */

function pushLabel(name) {
	if(findLabel(name)) return false;
	labelIndex[labelPtr++] = name + "|";
	return true;
}

/*
 *  findLabel() - Returns true if label exists.
 *
 */

function findLabel(name) {
	for(var m = 0; m < labelIndex.length; m++) {
		var nameAndAddr = labelIndex[m].split("|");
		if(name == nameAndAddr[0]) {
			return true;
		}
	}
	return false;
}

/*
 *  setLabelPC() - Associates label with address
 *
 */

function setLabelPC(name, addr) {
	for(var i = 0; i < labelIndex.length; i++) {
		var nameAndAddr = labelIndex[i].split("|");
		if(name == nameAndAddr[0]) {
			labelIndex[i] = name + "|" + addr;
			return true;
		}
	}
	return false;
}

/*
 *  getLabelPC() - Get address associated with label
 *
 */

function getLabelPC(name) {
	for(var i = 0; i < labelIndex.length; i++) {
		var nameAndAddr = labelIndex[i].split("|");
		if(name == nameAndAddr[0]) {
			return(nameAndAddr[1]);
		}
	}
	return -1;
}

/*
 *  compileLine()
 *
 *  Compiles one line of code.  Returns true if it compiled successfully,
 *  false otherwise.
 */

function compileLine(input, lineno) {
	var command = "";
	var param;
	input = input
		.replace(new RegExp(/^(.*?);.*/), "$1") // remove comments
		.replace(new RegExp(/^\s+/), "") // trim line
		.replace(new RegExp(/\s+$/), "");

	// Find command or label
	if(input.match(new RegExp(/^\w+:/))) {
		var label = input.replace(new RegExp(/(^\w+):.*$/), "$1");
		if(input.match(new RegExp(/^\w+:[\s]*\w+.*$/))) {
			input = input.replace(new RegExp(/^\w+:[\s]*(.*)$/), "$1");
			command = input.replace(new RegExp(/^(\w+).*$/), "$1");
		} else {
			command = "";
		}
	} else {
		command = input.replace(new RegExp(/^(\w+).*$/), "$1");
	}

	// Blank line?  Return.
	if(command == "") return true;

	command = command.toUpperCase();

	if(input.match(/^\*[\s]*=[\s]*[\$]?[0-9a-f]*$/)) {
		// equ spotted
		var addr;
		param = input.replace(new RegExp(/^[\s]*\*[\s]*=[\s]*/), "");
		if(param[0] == "$") {
			param = param.replace(new RegExp(/^\$/), "");
			addr = parseInt(param, 16);
		} else {
			addr = parseInt(param, 10);
		}
		if((addr < 0) || (addr > 0xffff)) {
			message("Unable to relocate code outside 64k memory");
			return false;
		}
		defaultCodePC = addr;
		return true;
	}

	if(input.match(/^\w+\s+.*?$/)) {
		param = input.replace(new RegExp(/^\w+\s+(.*?)/), "$1");
	} else {
		if(input.match(/^\w+$/)) {
			param = "";
		} else {
			return false;
		}
	}

	param = param.replace(/[ ]/g, "");

	if(command == "DCB") return DCB(param);

	for(var o = 0; o < Opcodes.length; o++) {
		if(Opcodes[o][0] == command) {
			if(checkSingle(param, Opcodes[o][10])) return true;
			if(checkImmediate(param, Opcodes[o][1])) return true;
			if(checkZeroPage(param, Opcodes[o][2])) return true;
			if(checkZeroPageX(param, Opcodes[o][3])) return true;
			if(checkZeroPageY(param, Opcodes[o][4])) return true;
			if(checkAbsoluteX(param, Opcodes[o][6])) return true;
			if(checkAbsoluteY(param, Opcodes[o][7])) return true;
			if(checkIndirectX(param, Opcodes[o][8])) return true;
			if(checkIndirectY(param, Opcodes[o][9])) return true;
			if(checkAbsolute(param, Opcodes[o][5])) return true;
			if(checkBranch(param, Opcodes[o][11])) return true;
		}
	}
	return false; // Unknown opcode
}

/*****************************************************************************
 ****************************************************************************/

function DCB(param) {
	var values = param.split(",");
	if(values.length == 0) return false;
	for(var v = 0; v < values.length; v++) {
		var str = values[v];
		if(str != undefined && str != null && str.length > 0) {
			var ch = str.substring(0, 1);
			if(ch == "$") {
				number = parseInt(str.replace(/^\$/, ""), 16);
				pushByte(number);
			} else if(ch >= "0" && ch <= "9") {
				var number = parseInt(str, 10);
				pushByte(number);
			} else {
				return false;
			}
		}
	}
	return true;
}

/*
 *  checkBranch() - Commom branch function for all branches (BCC, BCS, BEQ, BNE..)
 *
 */

function checkBranch(param, opcode) {
	if(opcode == 0x00) return false;

	var addr = -1;
	if(param.match(/\w+/))
		addr = getLabelPC(param);
	if(addr == -1) {
		pushWord(0x00);
		return false;
	}
	pushByte(opcode);
	if(addr < (defaultCodePC - 0x600)) { // Backwards?
		pushByte((0xff - ((defaultCodePC - 0x600) - addr)) & 0xff);
		return true;
	}
	pushByte((addr - (defaultCodePC - 0x600) - 1) & 0xff);
	return true;
}

/*
 * checkImmediate() - Check if param is immediate and push value
 * 
 */

function checkImmediate(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(new RegExp(/^#\$[0-9a-f]{1,2}$/i))) {
		pushByte(opcode);
		value = parseInt(param.replace(/^#\$/, ""), 16);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	if(param.match(new RegExp(/^#[0-9]{1,3}$/i))) {
		pushByte(opcode);
		var value = parseInt(param.replace(/^#/, ""), 10);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	// Label lo/hi
	if(param.match(new RegExp(/^#[<>]\w+$/))) {
		var label = param.replace(new RegExp(/^#[<>](\w+)$/), "$1");
		var hilo = param.replace(new RegExp(/^#([<>]).*$/), "$1");
		pushByte(opcode);
		if(findLabel(label)) {
			var addr = getLabelPC(label);
			switch(hilo) {
				case ">":
					pushByte((addr >> 8) & 0xff);
					return true;
					break;
				case "<":
					pushByte(addr & 0xff);
					return true;
					break;
				default:
					return false;
					break;
			}
		} else {
			pushByte(0x00);
			return true;
		}
	}
	return false;
}

/*
 * checkIndirectX() - Check if param is indirect X and push value
 * 
 */

function checkIndirectX(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\(\$[0-9a-f]{1,2},X\)$/i)) {
		pushByte(opcode);
		var value = param.replace(new RegExp(/^\(\$([0-9a-f]{1,2}).*$/i), "$1");
		if(value < 0 || value > 255) return false;
		pushByte(parseInt(value, 16));
		return true;
	}
	return false;
}

/*
 * checkIndirectY() - Check if param is indirect Y and push value
 * 
 */

function checkIndirectY(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\(\$[0-9a-f]{1,2}\),Y$/i)) {
		pushByte(opcode);
		var value = param.replace(new RegExp(/^\([\$]([0-9a-f]{1,2}).*$/i), "$1");
		if(value < 0 || value > 255) return false;
		pushByte(parseInt(value, 16));
		return true;
	}
	return false;
}

/*
 *  checkSingle() - Single-byte opcodes
 *
 */

function checkSingle(param, opcode) {
	if(opcode == 0x00) return false;
	if(param != "") return false;
	pushByte(opcode);
	return true;
}

/*
 *  checkZeroaPage() - Check if param is ZP and push value
 *
 */

function checkZeroPage(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\$[0-9a-f]{1,2}$/i)) {
		pushByte(opcode);
		var value = parseInt(param.replace(/^\$/, ""), 16);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	if(param.match(/^[0-9]{1,3}$/i)) {
		pushByte(opcode);
		var value = parseInt(param, 10);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	return false;
}

/*
 *  checkAbsoluteX() - Check if param is ABSX and push value
 *
 */

function checkAbsoluteX(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\$[0-9a-f]{3,4},X$/i)) {
		pushByte(opcode);
		var number = param.replace(new RegExp(/^\$([0-9a-f]*),X/i), "$1");
		var value = parseInt(number, 16);
		if(value < 0 || value > 0xffff) return false;
		pushWord(value);
		return true;
	}

	if(param.match(/^\w+,X$/i)) {
		param = param.replace(new RegExp(/,X$/i), "");
		pushByte(opcode);
		if(findLabel(param)) {
			var addr = getLabelPC(param);
			if(addr < 0 || addr > 0xffff) return false;
			pushWord(addr);
			return true;
		} else {
			pushWord(0x1234);
			return true;
		}
	}

	return false;
}

/*
 *  checkAbsoluteY() - Check if param is ABSY and push value
 *
 */

function checkAbsoluteY(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\$[0-9a-f]{3,4},Y$/i)) {
		pushByte(opcode);
		var number = param.replace(new RegExp(/^\$([0-9a-f]*),Y/i), "$1");
		var value = parseInt(number, 16);
		if(value < 0 || value > 0xffff) return false;
		pushWord(value);
		return true;
	}

	// it could be a label too..

	if(param.match(/^\w+,Y$/i)) {
		param = param.replace(new RegExp(/,Y$/i), "");
		pushByte(opcode);
		if(findLabel(param)) {
			var addr = getLabelPC(param);
			if(addr < 0 || addr > 0xffff) return false;
			pushWord(addr);
			return true;
		} else {
			pushWord(0x1234);
			return true;
		}
	}
	return false;
}

/*
 *  checkZeroPageX() - Check if param is ZPX and push value
 *
 */

function checkZeroPageX(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\$[0-9a-f]{1,2},X/i)) {
		pushByte(opcode);
		var number = param.replace(new RegExp(/^\$([0-9a-f]{1,2}),X/i), "$1");
		var value = parseInt(number, 16);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	if(param.match(/^[0-9]{1,3},X/i)) {
		pushByte(opcode);
		var number = param.replace(new RegExp(/^([0-9]{1,3}),X/i), "$1");
		var value = parseInt(number, 10);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	return false;
}

function checkZeroPageY(param, opcode) {
	if(opcode == 0x00) return false;
	if(param.match(/^\$[0-9a-f]{1,2},Y/i)) {
		pushByte(opcode);
		number = param.replace(new RegExp(/^\$([0-9a-f]{1,2}),Y/i), "$1");
		value = parseInt(number, 16);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	if(param.match(/^[0-9]{1,3},Y/i)) {
		pushByte(opcode);
		number = param.replace(new RegExp(/^([0-9]{1,3}),Y/i), "$1");
		value = parseInt(number, 10);
		if(value < 0 || value > 255) return false;
		pushByte(value);
		return true;
	}
	return false;
}

/*
 *  checkAbsolute() - Check if param is ABS and push value
 *
 */

function checkAbsolute(param, opcode) {
	if(opcode == 0x00) return false;
	pushByte(opcode);
	if(param.match(/^\$[0-9a-f]{3,4}$/i)) {
		var value = parseInt(param.replace(/^\$/, ""), 16);
		if(value < 0 || value > 0xffff) return false;
		pushWord(value);
		return true;
	}
	// it could be a label too..
	if(param.match(/^\w+$/)) {
		if(findLabel(param)) {
			var addr = (getLabelPC(param));
			if(addr < 0 || addr > 0xffff) return false;
			pushWord(addr);
			return true;
		} else {
			pushWord(0x1234);
			return true;
		}
	}
	return false;
}

/*****************************************************************************
 ****************************************************************************/

/*
 *  stackPush() - Push byte to stack
 *
 */

function stackPush(value) {
	if(regSP >= 0) {
		regSP--;
		memory[(regSP & 0xff) + 0x100] = value & 0xff;
	} else {
		message("Stack full: " + regSP);
		codeRunning = false;
	}
}

/*****************************************************************************
 ****************************************************************************/

/*
 *  stackPop() - Pop byte from stack
 *
 */

function stackPop() {
	if(regSP < 0x100) {
		var value = memory[regSP + 0x100];
		regSP++;
		return value;
	} else {
		message("Stack empty");
		codeRunning = false;
		return 0;
	}
}

/*
 * pushByte() - Push byte to compiledCode variable
 *
 */

function pushByte(value) {
	memory[defaultCodePC] = value & 0xff;
	defaultCodePC++;
	codeLen++;
}

/*
 * pushWord() - Push a word using pushByte twice
 *
 */

function pushWord(value) {
	pushByte(value & 0xff);
	pushByte((value >> 8) & 0xff);
}

/*
 * popByte() - Pops a byte
 *
 */

function popByte() {
	return(memory[regPC++] & 0xff);
}

/*
 * popWord() - Pops a word using popByte() twice
 *
 */

function popWord() {
	return popByte() + (popByte() << 8);
}

/*
 * memStoreByte() - Poke a byte, don't touch any registers
 *
 */

function memStoreByte(addr, value) {
	memory[addr] = (value & 0xff);
	if((addr >= 0x200) && (addr <= 0x5ff))
		display[addr - 0x200] = palette[memory[addr] & 0x0f];
}

/*
 * memStoreByte() - Peek a byte, don't touch any registers
 *
 */

function memReadByte(addr) {
	if(addr == 0xfe) return Math.floor(Math.random() * 256);
	return memory[addr];
}

function addr2hex(addr) {
	return num2hex((addr >> 8) & 0xff) + num2hex(addr & 0xff);
}

function num2hex(nr) {
	var str = "0123456789abcdef";
	var hi = ((nr & 0xf0) >> 4);
	var lo = (nr & 15);
	return str.substring(hi, hi + 1) + str.substring(lo, lo + 1);
}


/*
 *  runBinary() - Executes the compiled code
 *
 */

function runBinary() {
	if(codeRunning) {
		codeRunning = false;
	} else {
		//reset();
		codeRunning = true;
		//execute();
	}
}

/*
 *  readZeroPage() - Get value from ZP
 *
 */

function jumpBranch(offset) {
	if(offset > 0x7f)
		regPC = (regPC - (0x100 - offset));
	else
		regPC = (regPC + offset);
}

function doCompare(reg, val) {
	if((reg + val) > 0xff) regP |= 1;
	else regP &= 0xfe;
	val = (reg - val);
	//  if( reg+0x100-val > 0xff ) regP |= 1; else regP &= 0xfe;
	//  val = reg+0x100-val;
	if(val) regP &= 0xfd;
	else regP |= 0x02;
	if(val & 0x80) regP |= 0x80;
	else regP &= 0x7f;
}

function testSBC(value) {
	var vflag;
	if((regA ^ value) & 0x80) vflag = 1;
	else vflag = 0;

	if(regP & 8) {
		var tmp = 0xf + (regA & 0xf) - (value & 0xf) + (regP & 1);
		if(tmp < 0x10) {
			w = 0;
			tmp -= 6;
		} else {
			w = 0x10;
			tmp -= 0x10;
		}
		w += 0xf0 + (regA & 0xf0) - (value & 0xf0);
		if(w < 0x100) {
			regP &= 0xfe;
			if((regP & 0xbf) && w < 0x80) regP &= 0xbf;
			w -= 0x60;
		} else {
			regP |= 1;
			if((regP & 0xbf) && w >= 0x180) regP &= 0xbf;
		}
		w += tmp;
	} else {
		var w = 0xff + regA - value + (regP & 1);
		if(w < 0x100) {
			regP &= 0xfe;
			if((regP & 0xbf) && w < 0x80) regP &= 0xbf;
		} else {
			regP |= 1;
			if((regP & 0xbf) && w >= 0x180) regP &= 0xbf;
		}
	}
	regA = w & 0xff;
	if(regA) regP &= 0xfd;
	else regP |= 0x02;
	if(regA & 0x80) regP |= 0x80;
	else regP &= 0x7f;
}

function testADC(value) {
	if((regA ^ value) & 0x80) regP &= 0xbf;
	else regP |= 0x40;

	if(regP & 8) {
		var tmp = (regA & 0xf) + (value & 0xf) + (regP & 1);
		if(tmp >= 10) tmp = 0x10 | ((tmp + 6) & 0xf);
		
		tmp += (regA & 0xf0) + (value & 0xf0);
		if(tmp >= 160) {
			regP |= 1;
			if((regP & 0xbf) && tmp >= 0x180) regP &= 0xbf;
			tmp += 0x60;
		} else {
			regP &= 0xfe;
			if((regP & 0xbf) && tmp < 0x80) regP &= 0xbf;
		}
	} else {
		var tmp = regA + value + (regP & 1);
		if(tmp >= 0x100) {
			regP |= 1;
			if((regP & 0xbf) && tmp >= 0x180) regP &= 0xbf;
		} else {
			regP &= 0xfe;
			if((regP & 0xbf) && tmp < 0x80) regP &= 0xbf;
		}
	}
	regA = tmp & 0xff;
	if(regA) regP &= 0xfd;
	else regP |= 0x02;
	if(regA & 0x80) regP |= 0x80;
	else regP &= 0x7f;
}

function multiexecute() {
	for(var w = 0; w < 128; w++) execute();
}

/*
 *  execute() - Executes one instruction.
 *              This is the main part of the CPU emulator.
 *
 */

function execute() {
	if(!codeRunning) return;

	var opcode = popByte();
	// message( "PC=" + addr2hex(regPC-1) + " opcode=" + opcode + " X="+regX + " Y=" + regY + " A=" + regA );
	switch(opcode) {
		case 0x00: // BRK implied
			codeRunning = false;
			break;
		case 0x01: // ORA INDX
			var addr = popByte() + regX;
			var value = memReadByte(addr) + (memReadByte(addr + 1) << 8);
			regA |= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x05: // ORA ZP
			var zp = popByte();
			regA |= memReadByte(zp);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x06: // ASL ZP
			var zp = popByte();
			var value = memReadByte(zp);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			memStoreByte(zp, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x08: // PHP
			stackPush(regP);
			break;
		case 0x09: // ORA IMM
			regA |= popByte();
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x0a: // ASL IMPL
			regP = (regP & 0xfe) | ((regA >> 7) & 1);
			regA = regA << 1;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x0d: // ORA ABS
			regA |= memReadByte(popWord());
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x0e: // ASL ABS
			var addr = popWord();
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 2;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x10: // BPL
			var offset = popByte();
			if((regP & 0x80) == 0) jumpBranch(offset);
			break;
		case 0x11: // ORA INDY
			var zp = popByte();
			var value = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			regA |= memReadByte(value);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x15: // ORA ZPX
			var addr = (popByte() + regX) & 0xff;
			regA |= memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x16: // ASL ZPX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x18: // CLC
			regP &= 0xfe;
			break;
		case 0x19: // ORA ABSY
			var addr = popWord() + regY;
			regA |= memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x1d: // ORA ABSX
			var addr = popWord() + regX;
			regA |= memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x1e: // ASL ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x20: // JSR ABS
			var addr = popWord();
			var currAddr = regPC - 1;
			stackPush(((currAddr >> 8) & 0xff));
			stackPush((currAddr & 0xff));
			regPC = addr;
			break;
		case 0x21: // AND INDX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr) + (memReadByte(addr + 1) << 8);
			regA &= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x24: // BIT ZP
			var zp = popByte();
			var value = memReadByte(zp);
			if(value & regA) regP &= 0xfd;
			else regP |= 0x02;
			regP = (regP & 0x3f) | (value & 0xc0);
			break;
		case 0x25: // AND ZP
			var zp = popByte();
			regA &= memReadByte(zp);
			if(regA) regP &= 0xfd;
			else regP |= 2;
			if(regA & 0x80) regP &= 0x80;
			else regP &= 0x7f;
			break;
		case 0x26: // ROL ZP
			var sf = (regP & 1);
			var addr = popByte();
			var value = memReadByte(addr); //  & regA;  -- Thanks DMSC ;)
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			value |= sf;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x28: // PLP
			regP = stackPop() | 0x20;
			break;
		case 0x29: // AND IMM
			regA &= popByte();
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x2a: // ROL A
			var sf = (regP & 1);
			regP = (regP & 0xfe) | ((regA >> 7) & 1);
			regA = regA << 1;
			regA |= sf;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x2c: // BIT ABS
			var value = memReadByte(popWord());
			if(value & regA) regP &= 0xfd;
			else regP |= 0x02;
			regP = (regP & 0x3f) | (value & 0xc0);
			break;
		case 0x2d: // AND ABS
			var value = memReadByte(popWord());
			regA &= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x2e: // ROL ABS
			var sf = regP & 1;
			var addr = popWord();
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			value |= sf;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x30: // BMI
			var offset = popByte();
			if(regP & 0x80) jumpBranch(offset);
			break;
		case 0x31: // AND INDY
			var zp = popByte();
			var value = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			regA &= memReadByte(value);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x35: // AND INDX
			var zp = popByte();
			var value = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regX;
			regA &= memReadByte(value);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x36: // ROL ZPX
			var sf = regP & 1;
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			value |= sf;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x38: // SEC
			regP |= 1;
			break;
		case 0x39: // AND ABSY
			var addr = popWord() + regY;
			var value = memReadByte(addr);
			regA &= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x3d: // AND ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			regA &= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x3e: // ROL ABSX
			var sf = regP & 1;
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | ((value >> 7) & 1);
			value = value << 1;
			value |= sf;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x40: // RTI (unsupported, =NOP)
			break;
		case 0x41: // EOR INDX
			var zp = (popByte() + regX) & 0xff;
			var value = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			regA ^= memReadByte(value);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x45: // EOR ZPX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regA ^= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x46: // LSR ZP
			var addr = popByte() & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			memStoreByte(addr, value);
			if(value != 0) regP &= 0xfd;
			else regP |= 2;
			if((value & 0x80) == 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x48: // PHA
			stackPush(regA);
			break;
		case 0x49: // EOR IMM
			regA ^= popByte();
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x4a: // LSR
			regP = (regP & 0xfe) | (regA & 1);
			regA = regA >> 1;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x4c: // JMP abs
			regPC = popWord();
			break;
		case 0x4d: // EOR abs
			var addr = popWord();
			var value = memReadByte(addr);
			regA ^= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x4e: // LSR abs
			var addr = popWord();
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x50: // BVC (on overflow clear)
			var offset = popByte();
			if((regP & 0x40) == 0) jumpBranch(offset);
			break;
		case 0x51: // EOR INDY
			var zp = popByte();
			var value = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			regA ^= memReadByte(value);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x55: // EOR ZPX
			var addr = (popByte() + regX) & 0xff;
			regA ^= memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x56: // LSR ZPX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x58: // CLI (does nothing)
			break;
		case 0x59: // EOR ABSY
			var addr = popWord() + regY;
			var value = memReadByte(addr);
			regA ^= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x5d: // EOR ABSX
			var addr = popWord() + regX;
			value = memReadByte(addr);
			regA ^= value;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x5e: // LSR ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x60: // RTS
			regPC = (stackPop() + 1) | (stackPop() << 8);
			break;
		case 0x61: // ADC INDX
			var zp = (popByte() + regX) & 0xff;
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			var value = memReadByte(addr);
			testADC(value);
			break;
		case 0x65: // ADC ZP
			var addr = popByte();
			var value = memReadByte(addr);
			testADC(value);
			break;
		case 0x66: // ROR ZP
			var sf = regP & 1;
			var addr = popByte();
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			if(sf) value |= 0x80;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x68: // PLA
			regA = stackPop();
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x69: // ADC IMM
			var value = popByte();
			testADC(value);
			break;
		case 0x6a: // ROR A
			var sf = regP & 1;
			regP = (regP & 0xfe) | (regA & 1);
			regA = regA >> 1;
			if(sf) regA |= 0x80;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x6c: // JMP INDIR
			//      regPC = memReadByte(popByte()) + (memReadByte(popByte())<<8);
			break;
		case 0x6d: // ADC ABS
			var addr = popWord();
			var value = memReadByte(addr);
			testADC(value);
			break;
		case 0x6e: // ROR ABS
			var sf = regP & 1;
			var addr = popWord();
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			if(sf) value |= 0x80;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x70: // BVS (branch on overflow set)
			var offset = popByte();
			if(regP & 0x40) jumpBranch(offset);
			break;
		case 0x71: // ADC INY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			var value = memReadByte(addr + regY);
			testADC(value);
			break;
		case 0x75: // ADC ZPX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			testADC(value);
			break;
		case 0x76: // ROR ZPX
			var sf = (regP & 1);
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			if(sf) value |= 0x80;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x78: // SEI (does nothing)
			break;
		case 0x79: // ADC ABSY
			var addr = popWord();
			var value = memReadByte(addr + regY);
			testADC(value);
			break;
		case 0x7d: // ADC ABSX
			var addr = popWord();
			var value = memReadByte(addr + regX);
			testADC(value);
			break;
		case 0x7e: // ROR ABSX
			var sf = regP & 1;
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			value = value >> 1;
			if(value) value |= 0x80;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x81: // STA INDX
			var zp = (popByte() + regX) & 0xff;
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			memStoreByte(addr, regA);
			break;
		case 0x84: // STY ZP
			memStoreByte(popByte(), regY);
			break;
		case 0x85: // STA ZP
			memStoreByte(popByte(), regA);
			break;
		case 0x86: // STX ZP
			memStoreByte(popByte(), regX);
			break;
		case 0x88: // DEY (1 byte)
			regY = (regY - 1) & 0xff;
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x8a: // TXA (1 byte);
			regA = regX & 0xff;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x8c: // STY abs
			memStoreByte(popWord(), regY);
			break;
		case 0x8d: // STA ABS (3 bytes)
			memStoreByte(popWord(), regA);
			break;
		case 0x8e: // STX abs
			memStoreByte(popWord(), regX);
			break;
		case 0x90: // BCC (branch on carry clear)
			var offset = popByte();
			if((regP & 1) == 0) jumpBranch(offset);
			break;
		case 0x91: // STA INDY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			memStoreByte(addr, regA);
			break;
		case 0x94: // STY ZPX
			memStoreByte(popByte() + regX, regY);
			break;
		case 0x95: // STA ZPX
			memStoreByte(popByte() + regX, regA);
			break;
		case 0x96: // STX ZPY
			memStoreByte(popByte() + regY, regX);
			break;
		case 0x98: // TYA
			regA = regY & 0xff;
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0x99: // STA ABSY
			memStoreByte(popWord() + regY, regA);
			break;
		case 0x9a: // TXS
			regSP = regX & 0xff;
			break;
		case 0x9d: // STA ABSX
			var addr = popWord();
			memStoreByte(addr + regX, regA);
			break;
		case 0xa0: // LDY IMM
			regY = popByte();
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa1: // LDA INDX
			var zp = (popByte() + regX) & 0xff;
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			regA = memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa2: // LDX IMM
			regX = popByte();
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa4: // LDY ZP
			regY = memReadByte(popByte());
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa5: // LDA ZP
			regA = memReadByte(popByte());
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa6: // LDX ZP
			regX = memReadByte(popByte());
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa8: // TAY
			regY = regA & 0xff;
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xa9: // LDA IMM
			regA = popByte();
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xaa: // TAX
			regX = regA & 0xff;
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xac: // LDY ABS
			regY = memReadByte(popWord());
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xad: // LDA ABS
			regA = memReadByte(popWord());
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xae: // LDX ABS
			regX = memReadByte(popWord());
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xb0: // BCS
			var offset = popByte();
			if(regP & 1) jumpBranch(offset);
			break;
		case 0xb1: // LDA INDY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			regA = memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xb4: // LDY ZPX
			regY = memReadByte(popByte() + regX);
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xb5: // LDA ZPX
			regA = memReadByte((popByte() + regX) & 0xff);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xb6: // LDX ZPY
			regX = memReadByte(popByte() + regY);
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xb8: // CLV
			regP &= 0xbf;
			break;
		case 0xb9: // LDA ABSY
			var addr = popWord() + regY;
			regA = memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xba: // TSX
			regX = regSP & 0xff;
			break;
		case 0xbc: // LDY ABSX
			var addr = popWord() + regX;
			regY = memReadByte(addr);
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xbd: // LDA ABSX
			var addr = popWord() + regX;
			regA = memReadByte(addr);
			if(regA) regP &= 0xfd;
			else regP |= 0x02;
			if(regA & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xbe: // LDX ABSY
			var addr = popWord() + regY;
			regX = memReadByte(addr);
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xc0: // CPY IMM
			var value = popByte();
			if((regY + value) > 0xff) regP |= 1;
			else regP &= 0xfe;
			var ov = value;
			var value = (regY - value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xc1: // CMP INDY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			var value = memReadByte(addr);
			doCompare(regA, value);
			break;
		case 0xc4: // CPY ZP
			var value = memReadByte(popByte());
			doCompare(regY, value);
			break;
		case 0xc5: // CMP ZP
			var value = memReadByte(popByte());
			doCompare(regA, value);
			break;
		case 0xc6: // DEC ZP
			var zp = popByte();
			var value = memReadByte(zp);
			--value;
			memStoreByte(zp, value & 0xff);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xc8: // INY
			regY = (regY + 1) & 0xff;
			if(regY) regP &= 0xfd;
			else regP |= 0x02;
			if(regY & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xc9: // CMP IMM
			var value = popByte();
			doCompare(regA, value);
			break;
		case 0xca: // DEX
			regX = (regX - 1) & 0xff;
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xcc: // CPY ABS
			var value = memReadByte(popWord());
			doCompare(regY, value);
			break;
		case 0xcd: // CMP ABS
			value = memReadByte(popWord());
			doCompare(regA, value);
			break;
		case 0xce: // DEC ABS
			var addr = popWord();
			var value = memReadByte(addr);
			--value;
			value = value & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xd0: // BNE
			var offset = popByte();
			//      if( (regP&2)==0 ) { oldPC = regPC; jumpBranch( offset ); message( "Jumping from " + addr2hex(oldPC) + " to " + addr2hex(regPC) ); } else { message( "NOT jumping!" ); }
			if((regP & 2) == 0) jumpBranch(offset);
			break;
		case 0xd1: // CMP INDY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8) + regY;
			var value = memReadByte(addr);
			doCompare(regA, value);
			break;
		case 0xd5: // CMP ZPX
			var value = memReadByte(popByte() + regX);
			doCompare(regA, value);
			break;
		case 0xd6: // DEC ZPX
			var addr = popByte() + regX;
			var value = memReadByte(addr);
			--value;
			value = value & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xd8: // CLD (CLear Decimal)
			regP &= 0xf7;
			break;
		case 0xd9: // CMP ABSY
			var addr = popWord() + regY;
			var value = memReadByte(addr);
			doCompare(regA, value);
			break;
		case 0xdd: // CMP ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			doCompare(regA, value);
			break;
		case 0xde: // DEC ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			--value;
			value = value & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xe0: // CPX IMM
			var value = popByte();
			doCompare(regX, value);
			break;
		case 0xe1: // SBC INDX
			var zp = (popByte() + regX) & 0xff;
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			var value = memReadByte(addr);
			testSBC(value);
			break;
		case 0xe4: // CPX ZP
			var value = memReadByte(popByte());
			doCompare(regX, value);
			break;
		case 0xe5: // SBC ZP
			var addr = popByte();
			var value = memReadByte(addr);
			testSBC(value);
			break;
		case 0xe6: // INC ZP
			var zp = popByte();
			var value = memReadByte(zp);
			++value;
			value = (value) & 0xff;
			memStoreByte(zp, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xe8: // INX
			regX = (regX + 1) & 0xff;
			if(regX) regP &= 0xfd;
			else regP |= 0x02;
			if(regX & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xe9: // SBC IMM
			var value = popByte();
			testSBC(value);
			break;
		case 0xea: // NOP
			break;
		case 0xec: // CPX ABS
			var value = memReadByte(popWord());
			doCompare(regX, value);
			break;
		case 0xed: // SBC ABS
			var addr = popWord();
			var value = memReadByte(addr);
			testSBC(value);
			break;
		case 0xee: // INC ABS
			var addr = popWord();
			var value = memReadByte(addr);
			++value;
			value = (value) & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xf0: // BEQ
			var offset = popByte();
			if(regP & 2) jumpBranch(offset);
			break;
		case 0xf1: // SBC INDY
			var zp = popByte();
			var addr = memReadByte(zp) + (memReadByte(zp + 1) << 8);
			var value = memReadByte(addr + regY);
			testSBC(value);
			break;
		case 0xf5: // SBC ZPX
			var addr = (popByte() + regX) & 0xff;
			var value = memReadByte(addr);
			regP = (regP & 0xfe) | (value & 1);
			testSBC(value);
			break;
		case 0xf6: // INC ZPX
			var addr = popByte() + regX;
			var value = memReadByte(addr);
			++value;
			value = value & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		case 0xf8: // SED
			regP |= 8;
			break;
		case 0xf9: // SBC ABSY
			var addr = popWord();
			var value = memReadByte(addr + regY);
			testSBC(value);
			break;
		case 0xfd: // SBC ABSX
			var addr = popWord();
			var value = memReadByte(addr + regX);
			testSBC(value);
			break;
		case 0xfe: // INC ABSX
			var addr = popWord() + regX;
			var value = memReadByte(addr);
			++value;
			value = value & 0xff;
			memStoreByte(addr, value);
			if(value) regP &= 0xfd;
			else regP |= 0x02;
			if(value & 0x80) regP |= 0x80;
			else regP &= 0x7f;
			break;
		default:
			message("Address $" + addr2hex(regPC) + " - unknown opcode " + opcode);
			codeRunning = false;
			break;
	}

	if((regPC == 0) || (!codeRunning)) {
		message("Program end at PC=$" + addr2hex(regPC - 1));
		codeRunning = false;
		//updateDisplayFull();
	}
}

/*
 *  updatePixelDisplay() - Updates the display at one pixel position
 *
 */

function updateDisplayPixel(addr) {
	display[addr - 0x200] = palette[memory[addr] & 0x0f];
}


/*
 *  updateDisplayFull() - Simply redraws the entire display according to memory
 *  The colors are supposed to be identical with the C64's palette.
 *
 */

function updateDisplayFull() {
	for(var y = 0; y < 32; y++) {
		for(var x = 0; x < 32; x++) {
			updateDisplayPixel(((y << 5) + x) + 0x200);
		}
	}
}


// Sphere stuff
var console = new Console({
	logFileName: "~/assembler.log",
});
console.font = font;

console.defineObject("asm", null, {
	"load": compileCode,
	"reset": reset,
	"run": runBinary,
	"recompile": function() {
		compileCode(filename);
	},
	"hexdump": function() {
		var dump_text = "";
		for(var x = 0; x < codeLen; x++) {
			if(x % 16 == 0) {
				console.log(dump_text);
				dump_text = "";
			}
			if((x&15) == 0) {
				var n = (0x600+x);
				dump_text += num2hex(((n>>8)&0xff)) + num2hex((n&0xff)) + ": ";
			}
			dump_text += num2hex(memory[0x600+x]);
			if(x&1) dump_text += " ";
		}
		if((x&1)) dump_text += "-- [END]";
		console.log(dump_text);
	}
});

let pixels = new Uint8Array(32 * 32 * 4);
pixels.fill(255);
let pixelShape = new Shape(ShapeType.TriStrip, null, new VertexList([
	{ x: 0,  y: 0,  u: 0, v: 1 },
	{ x: 32, y: 0,  u: 1, v: 1 },
	{ x: 0,  y: 32, u: 0, v: 0 },
	{ x: 32, y: 32, u: 1, v: 0 },
]));
let transform = new Transform();

Dispatch.onRender(function(){
	for(var i = 0; i < 32*32; i++) {
		let offset = i * 4;
		pixels[offset]     = display[i][0]; //red
		pixels[offset + 1] = display[i][1]; //green
		pixels[offset + 2] = display[i][2]; //blue
	}
	transform.identity().scale(scale, scale);
	pixelShape.texture = new Texture(32, 32, pixels);
	pixelShape.draw(screen, transform);

	if(drawDebugInfo) {
		font.drawText(screen, 2, 0,
			"PC:" + addr2hex(regPC-1), Color.Black);
		font.drawText(screen, 2, Font.Default.height + 2,
			"X=" + regX, Color.Black);
		font.drawText(screen, 2, Font.Default.height * 2 + 2,
			"Y=" + regY, Color.Black);
		font.drawText(screen, 2, Font.Default.height * 3 + 2,
			"A=" + regA, Color.Black);
	}
});

Dispatch.onUpdate(function() {
	var key_pressed = Keyboard.Default.getKey();
	if(key_pressed == Key.Escape) Exit();
	if(key_pressed == Key.F1) codeRunning = !codeRunning;
	if(key_pressed == Key.F2) drawDebugInfo = !drawDebugInfo;
	if(codeRunning) multiexecute();

	// Store keycode in $ff
	if(key_pressed != null) {
		memStoreByte(0xff, Keyboard.Default.charOf(key_pressed,
			Keyboard.Default.isPressed(Key.LShift) || 
			Keyboard.Default.isPressed(Key.RShift)).charCodeAt(0)
		);
	}
});

compileCode("games/zookeeper.asm");
reset();
runBinary();
